from flask import Flask, render_template, request, redirect, url_for, jsonify
from flask_socketio import SocketIO, emit, join_room, leave_room
import secrets
from datetime import datetime

app = Flask(__name__)
app.config['SECRET_KEY'] = secrets.token_hex(16)
socketio = SocketIO(app, cors_allowed_origins="*")

# Room information
rooms = {}  # Format: {
    # room_id: {
    #     'password': 'xxx',
    #     'broadcaster': None,
    #     'viewers': {},  # {sid: {'display_name': 'xxx', 'permissions': {...}, 'banned': False}}
    #     'messages': [],
    #     'banned_users': set(),
    #     'muted_users': set(),
    #     'active_streams': {  # Track active stream types
    #         'camera': False,
    #         'screen': False,
    #         'mic': False
    #     }
    # }
# }

@app.route('/')
def index():
    return '''
    <!DOCTYPE html>
<html>
<head>
    <title>Live Broadcast</title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/socket.io/4.0.1/socket.io.js"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            min-height: 100vh;
            color: #fff;
        }
        
         .fullscreen-btn {
        position: absolute;
        right: 10px;
        top: 10px;
        background: rgba(0, 0, 0, 0.5);
        border: none;
        border-radius: 4px;
        padding: 8px;
        cursor: pointer;
        z-index: 20;
        transition: all 0.3s ease;
    }

    .fullscreen-btn:hover {
        background: rgba(0, 0, 0, 0.7);
    }

    /* Add fullscreen styles */
    .stream-container.fullscreen {
        position: fixed;
        top: 0;
        left: 0;
        width: 100vw;
        height: 100vh;
        z-index: 1000;
        background: black;
    }

    .stream-container.fullscreen .camera-video.full {
        width: 100vw;
        height: 100vh;
    }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .stream-container {
            position: relative;
            width: 100%;
            background: rgba(0, 0, 0, 0.2);
            aspect-ratio: 16/9;
        }

        .screen-video {
            position: absolute;
            width: 100%;
            height: 100%;
            object-fit: contain;
        }

        .camera-video {
            position: absolute;
            width: 25%;
            aspect-ratio: 16/9;
            right: 20px;
            bottom: 20px;
            border: 2px solid rgba(255, 255, 255, 0.2);
            border-radius: 8px;
            background: rgba(0, 0, 0, 0.3);
            transition: all 0.3s ease;
            z-index: 10;
        }

        .camera-video.full {
            width: 100%;
            height: 100%;
            right: 0;
            bottom: 0;
            border: none;
            border-radius: 0;
        }

        .camera-video:hover {
            transform: scale(1.05);
        }

        #chat-container::-webkit-scrollbar {
            width: 6px;
        }

        #chat-container::-webkit-scrollbar-track {
            background: rgba(255, 255, 255, 0.1);
        }

        #chat-container::-webkit-scrollbar-thumb {
            background: rgba(255, 255, 255, 0.3);
            border-radius: 3px;
        }

        .message-animation {
            animation: slideIn 0.3s ease-out;
        }

        @keyframes slideIn {
            from { transform: translateX(-10px); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
    </style>
</head>
<div class="container mx-auto px-4 py-8 max-w-8xl">
        <h1 class="text-4xl font-bold mb-8 text-center text-white">Live Broadcast</h1>
        
        <!-- Join Form -->
        <div id="join-container" class="max-w-md mx-auto glass-effect p-8 rounded-lg">
            <div class="space-y-4">
                <div>
                    <label class="block text-sm font-medium mb-1">Display Name</label>
                    <input type="text" id="display-name" class="w-full px-4 py-2 rounded bg-gray-800 border border-gray-700 focus:border-blue-500 focus:outline-none text-white" placeholder="Enter your name">
                </div>
                <div>
                    <label class="block text-sm font-medium mb-1">Room ID</label>
                    <input type="text" id="room-id" class="w-full px-4 py-2 rounded bg-gray-800 border border-gray-700 focus:border-blue-500 focus:outline-none text-white" placeholder="Enter room ID">
                </div>
                <div>
                    <label class="block text-sm font-medium mb-1">Password</label>
                    <input type="password" id="room-password" class="w-full px-4 py-2 rounded bg-gray-800 border border-gray-700 focus:border-blue-500 focus:outline-none text-white" placeholder="Enter password">
                </div>
                <div class="flex space-x-4">
                    <button onclick="createRoom()" class="flex-1 bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded transition-colors">Create Room</button>
                    <button onclick="joinRoom()" class="flex-1 bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-4 rounded transition-colors">Join Room</button>
                </div>
                <div id="join-error" class="hidden text-red-500 text-sm text-center"></div>
            </div>
        </div>

        <!-- Broadcast Container -->
        <div id="broadcast-container" class="hidden">
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- Video Section -->
                <div class="lg:col-span-2">
                    <div class="glass-effect rounded-lg overflow-hidden">
                        <div id="streams-container" class="stream-container">
    <button onclick="toggleFullscreen()" class="fullscreen-btn">
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2">
            <path d="M8 3H5a2 2 0 0 0-2 2v3m18 0V5a2 2 0 0 0-2-2h-3m0 18h3a2 2 0 0 0 2-2v-3M3 16v3a2 2 0 0 0 2 2h3"/>
        </svg>
    </button>
    <video id="screen-video" class="screen-video" autoplay playsinline></video>
    <video id="camera-video" class="camera-video" autoplay playsinline></video>
</div>
                        
                        <!-- Broadcaster Controls -->
                        <div id="broadcaster-controls" class="p-4 space-y-4 hidden">
                            <div class="flex flex-wrap gap-2">
                                <button onclick="toggleStream('camera')" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded flex items-center">
                                    <span id="camera-status" class="w-2 h-2 rounded-full bg-red-500 mr-2"></span>
                                    Camera
                                </button>
                                <button onclick="toggleStream('screen')" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded flex items-center">
                                    <span id="screen-status" class="w-2 h-2 rounded-full bg-red-500 mr-2"></span>
                                    Screen
                                </button>
                                <button onclick="toggleStream('mic')" class="bg-yellow-600 hover:bg-yellow-700 text-white px-4 py-2 rounded flex items-center">
                                    <span id="mic-status" class="w-2 h-2 rounded-full bg-red-500 mr-2"></span>
                                    Microphone
                                </button>
                            </div>
                        </div>
                        
                        <!-- Viewer Controls -->
                        <div id="viewer-controls" class="p-4 space-y-2 hidden">
                            <button id="request-share" onclick="requestSharePermission()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">Request to Share</button>
                        </div>
                    </div>
                </div>

                <!-- Chat Section -->
                <div class="space-y-4">
                    <!-- Users List -->
                    <div class="glass-effect rounded-lg p-4">
                        <h3 class="text-lg font-semibold mb-3">Participants</h3>
                        <div id="users-container" class="space-y-2"></div>
                    </div>

                    <!-- Chat Area -->
                    <div class="glass-effect rounded-lg p-4 flex flex-col h-96">
                        <div id="chat-container" class="flex-1 overflow-y-auto space-y-2 mb-4"></div>
                        <div class="flex space-x-2">
                            <input type="text" id="message" class="flex-1 px-4 py-2 rounded bg-gray-800 border border-gray-700 focus:border-blue-500 focus:outline-none text-white" placeholder="Type your message..." onkeypress="handleKeyPress(event)">
                            <button onclick="sendMessage()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">Send</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        const socket = io();
        let streams = {
            camera: null,
            screen: null,
            mic: null
        };
        let roomId = null;
        let isBroadcaster = false;
        let displayName = '';
        let peerConnections = {};

        function toggleFullscreen() {
        const container = document.getElementById('streams-container');
        
        if (!document.fullscreenElement) {
            container.requestFullscreen().catch(err => {
                console.error(`Error attempting to enable fullscreen: ${err.message}`);
            });
            container.classList.add('fullscreen');
        } else {
            document.exitFullscreen();
            container.classList.remove('fullscreen');
        }
    }

        
             async function createRoom() {
            displayName = document.getElementById('display-name').value.trim();
            roomId = document.getElementById('room-id').value.trim();
            const password = document.getElementById('room-password').value;
            
            if (!displayName || !roomId || !password) {
                showError('All fields are required');
                return;
            }

            isBroadcaster = true;
            
            socket.emit('create_room', {
                room_id: roomId,
                password: password,
                display_name: displayName
            });

            document.getElementById('broadcaster-controls').classList.remove('hidden');
        }
        
        async function toggleStream(type) {
    try {
        if (streams[type]) {
            // Stop the stream
            streams[type].getTracks().forEach(track => track.stop());
            streams[type] = null;
            
            if (type !== 'mic') {
                document.getElementById(`${type}-video`).classList.add('hidden');
                document.getElementById(`${type}-video`).srcObject = null;
            }
            
            document.getElementById(`${type}-status`).classList.remove('bg-green-500');
            document.getElementById(`${type}-status`).classList.add('bg-red-500');
        } else {
            // Start the stream
            let newStream;
            if (type === 'camera') {
                newStream = await navigator.mediaDevices.getUserMedia({
                    video: true,
                    audio: false
                });
                // Label the video track
                newStream.getVideoTracks()[0].label = 'camera';
            } else if (type === 'screen') {
                newStream = await navigator.mediaDevices.getDisplayMedia({
                    video: true,
                    audio: false
                });
                // Label the video track
                newStream.getVideoTracks()[0].label = 'screen';
            } else if (type === 'mic') {
                newStream = await navigator.mediaDevices.getUserMedia({
                    video: false,
                    audio: true
                });
            }
            
            streams[type] = newStream;
            
            if (type !== 'mic') {
                const videoElement = document.getElementById(`${type}-video`);
                videoElement.srcObject = newStream;
                videoElement.classList.remove('hidden');
            }
            
            document.getElementById(`${type}-status`).classList.remove('bg-red-500');
            document.getElementById(`${type}-status`).classList.add('bg-green-500');
        }
        
        // Update peer connections with new stream state
        await updatePeerConnections();
        
        // Notify server about stream status
        socket.emit('stream_status', {
            room: roomId,
            type: type,
            active: !!streams[type]
        });
        
    } catch (err) {
        showError(`Error toggling ${type}: ${err.message}`);
    }
}

        async function updatePeerConnections() {
    for (const [peerId, pc] of Object.entries(peerConnections)) {
        try {
            // Remove all existing tracks
            const senders = pc.getSenders();
            for (const sender of senders) {
                pc.removeTrack(sender);
            }
            
            // Add all active streams
            for (const [streamType, stream] of Object.entries(streams)) {
                if (stream) {
                    stream.getTracks().forEach(track => {
                        pc.addTrack(track, stream);
                    });
                }
            }
            
            // Create and send new offer
            const offer = await pc.createOffer();
            await pc.setLocalDescription(offer);
            socket.emit('offer', {
                offer: offer,
                to: peerId
            });
        } catch (err) {
            console.error('Error updating peer connection:', err);
        }
    }
}
        
        function requestSharePermission() {
            socket.emit('request_share', { room: roomId });
        }

        function updateUserList(users) {
            const usersContainer = document.getElementById('users-container');
            usersContainer.innerHTML = '';
            
            Object.entries(users).forEach(([sid, user]) => {
                const userElement = document.createElement('div');
                userElement.className = 'flex items-center justify-between p-2';
                
                const userInfo = document.createElement('div');
                userInfo.className = 'flex items-center space-x-2';
                userInfo.innerHTML = `
                    <div class="w-2 h-2 rounded-full ${user.banned ? 'bg-red-500' : 'bg-green-500'}"></div>
                    <span>${user.display_name}${user.isBroadcaster ? ' (Host)' : ''}</span>
                `;
                
                const controls = document.createElement('div');
                controls.className = 'space-x-2';
                
                if (isBroadcaster && !user.isBroadcaster) {
                    controls.innerHTML = `
                        <button onclick="kickUser('${sid}')" class="text-xs bg-red-600 hover:bg-red-700 text-white px-2 py-1 rounded">Kick</button>
                        <button onclick="toggleMute('${sid}')" class="text-xs bg-yellow-600 hover:bg-yellow-700 text-white px-2 py-1 rounded">${user.muted ? 'Unmute' : 'Mute'}</button>
                        <button onclick="banUser('${sid}')" class="text-xs bg-red-800 hover:bg-red-900 text-white px-2 py-1 rounded">Ban</button>
                    `;
                }
                
                userElement.appendChild(userInfo);
                userElement.appendChild(controls);
                usersContainer.appendChild(userElement);
            });
        }

        function kickUser(sid) {
            socket.emit('kick_user', { room: roomId, target_sid: sid });
        }

        function banUser(sid) {
            socket.emit('ban_user', { room: roomId, target_sid: sid });
        }

        function toggleMute(sid) {
            socket.emit('toggle_mute', { room: roomId, target_sid: sid });
        }

        // Add event listeners for user management responses
        socket.on('share_request_response', function(data) {
            if (data.approved) {
                // Handle approved share request
                startSharing();
            } else {
                showError('Share request denied');
            }
        });

        socket.on('kicked', function() {
            window.location.reload();
        });

        socket.on('banned', function() {
            showError('You have been banned from this room');
            window.location.reload();
        });

            function joinRoom() {
                displayName = document.getElementById('display-name').value.trim();
                roomId = document.getElementById('room-id').value.trim();
                const password = document.getElementById('room-password').value;
                
                if (!displayName || !roomId || !password) {
                    showError('All fields are required');
                    return;
                }
                
                socket.emit('join_room', {
                    room_id: roomId,
                    password: password,
                    display_name: displayName
                });
            }

            function showError(message) {
            const errorDiv = document.getElementById('join-error');
            errorDiv.textContent = message;
            errorDiv.classList.remove('hidden');
            setTimeout(() => {
                errorDiv.classList.add('hidden');
            }, 5000);
        }

            function handleKeyPress(event) {
                if (event.key === 'Enter') {
                    sendMessage();
                }
            }

            function sendMessage() {
                const messageInput = document.getElementById('message');
                const message = messageInput.value.trim();
                if (message) {
                    socket.emit('message', {
                        room: roomId,
                        message: message,
                        display_name: displayName
                    });
                    messageInput.value = '';
                }
            }

            function addMessageToChat(data) {
            const chatContainer = document.getElementById('chat-container');
            const messageElement = document.createElement('div');
            messageElement.className = `message-animation p-2 rounded ${data.type === 'system' ? 'bg-gray-800 italic' : 'glass-effect'}`;
            
            const time = new Date().toLocaleTimeString();
            
            if (data.type === 'system') {
                messageElement.innerHTML = `<span class="text-gray-400">${data.message}</span>`;
            } else {
                messageElement.innerHTML = `
                    <span class="text-gray-400 text-xs">${time}</span>
                    <span class="text-blue-400 font-medium">${data.display_name}</span>
                    <span class="text-gray-300 ml-2">${data.message}</span>
                `;
            }
            
            chatContainer.appendChild(messageElement);
            chatContainer.scrollTop = chatContainer.scrollHeight;
        }

       function updateStreamLayout() {
        const screenVideo = document.getElementById('screen-video');
        const cameraVideo = document.getElementById('camera-video');
        
        const isScreenActive = screenVideo.srcObject && !screenVideo.classList.contains('hidden');
        const isCameraActive = cameraVideo.srcObject && !cameraVideo.classList.contains('hidden');
        
        if (isScreenActive && isCameraActive) {
            cameraVideo.classList.remove('full');
            screenVideo.classList.remove('hidden');
        } else if (isCameraActive) {
            cameraVideo.classList.add('full');
            screenVideo.classList.add('hidden');
        } else if (isScreenActive) {
            screenVideo.classList.remove('hidden');
            cameraVideo.classList.add('hidden');
        } else {
            screenVideo.classList.add('hidden');
            cameraVideo.classList.add('hidden');
        }
    }


            function updateUserList(users) {
            const usersContainer = document.getElementById('users-container');
            usersContainer.innerHTML = '';
            Object.values(users).forEach(user => {
                const userElement = document.createElement('div');
                userElement.className = 'flex items-center space-x-2';
                userElement.innerHTML = `
                    <div class="w-2 h-2 rounded-full bg-green-500"></div>
                    <span>${user.display_name}${user.isBroadcaster ? ' <span class="text-blue-400 text-sm">(Host)</span>' : ''}</span>
                `;
                usersContainer.appendChild(userElement);
            });
        }

            socket.on('room_joined', function(data) {
                document.getElementById('join-container').classList.add('hidden');
                document.getElementById('broadcast-container').classList.remove('hidden');
                
                // Load previous messages
                data.messages.forEach(addMessageToChat);
                updateUserList(data.users);
            });

            socket.on('message', function(data) {
                addMessageToChat(data);
            });

            socket.on('user_list_updated', function(data) {
                updateUserList(data.users);
            });

            socket.on('room_error', function(data) {
                showError(data.message);
                if (localStream) {
                    localStream.getTracks().forEach(track => track.stop());
                    localStream = null;
                }
            });

            // WebRTC signaling
         socket.on('offer', async function(data) {
    if (!isBroadcaster) {
        try {
            let pc = peerConnections[data.from];
            
            if (!pc) {
                pc = new RTCPeerConnection({
                    iceServers: [
                        { urls: 'stun:stun.l.google.com:19302' }
                    ]
                });
                
                peerConnections[data.from] = pc;
                
                pc.onicecandidate = event => {
                    if (event.candidate) {
                        socket.emit('ice_candidate', {
                            candidate: event.candidate,
                            to: data.from
                        });
                    }
                };

                pc.ontrack = event => {
                    const track = event.track;
                    
                    if (track.kind === 'video') {
                        // Determine video element based on track label
                        const videoId = track.label === 'screen' ? 'screen-video' : 'camera-video';
                        const videoElement = document.getElementById(videoId);
                        
                        if (videoElement) {
                            if (!videoElement.srcObject) {
                                videoElement.srcObject = new MediaStream();
                            }
                            videoElement.srcObject.addTrack(track);
                            videoElement.classList.remove('hidden');
                            
                            track.onended = () => {
                                const stream = videoElement.srcObject;
                                stream.removeTrack(track);
                                if (stream.getTracks().length === 0) {
                                    videoElement.srcObject = null;
                                    videoElement.classList.add('hidden');
                                }
                                updateStreamLayout();
                            };
                            
                            updateStreamLayout();
                        }
                    } else if (track.kind === 'audio') {
                        const audioElement = new Audio();
                        audioElement.srcObject = new MediaStream([track]);
                        audioElement.play().catch(console.error);
                    }
                };
            }

            // Check if we already have a remote description
            if (pc.remoteDescription === null) {
                await pc.setRemoteDescription(new RTCSessionDescription(data.offer));
            }
            
            // Create and set local description only if we haven't already
            if (pc.localDescription === null) {
                const answer = await pc.createAnswer();
                await pc.setLocalDescription(answer);
                
                // Send answer back to broadcaster
                socket.emit('answer', {
                    answer: answer,
                    to: data.from
                });
            }

            // Process any pending ICE candidates
            if (pc.pendingCandidates && pc.pendingCandidates.length > 0) {
                for (const candidate of pc.pendingCandidates) {
                    await pc.addIceCandidate(new RTCIceCandidate(candidate));
                }
                pc.pendingCandidates = [];
            }

        } catch (err) {
            console.error('Error handling offer:', err);
        }
    }
});

socket.on('stream_status_changed', function(data) {
    updateStreamLayout();
});

           // Handle ICE candidates
socket.on('ice_candidate', async function(data) {
    try {
        const pc = peerConnections[data.from];
        if (pc) {
            if (pc.remoteDescription && pc.remoteDescription.type) {
                await pc.addIceCandidate(new RTCIceCandidate(data.candidate));
            } else {
                if (!pc.pendingCandidates) {
                    pc.pendingCandidates = [];
                }
                pc.pendingCandidates.push(data.candidate);
            }
        }
    } catch (err) {
        console.error('Error handling ICE candidate:', err);
    }
});



             // Modify viewer-side offer handling
        socket.on('offer', async function(data) {
    if (!isBroadcaster) {
        try {
            // Create or get existing peer connection
            let pc = peerConnections[data.from];
            if (!pc) {
                pc = new RTCPeerConnection();
                peerConnections[data.from] = pc;
                
                pc.onicecandidate = event => {
                    if (event.candidate) {
                        socket.emit('ice_candidate', {
                            candidate: event.candidate,
                            to: data.from
                        });
                    }
                };

                pc.ontrack = event => {
                    const track = event.track;
                    const streams = event.streams;
                    
                    if (track.kind === 'video') {
                        // Determine which video element to use based on track label
                        const videoId = track.label === 'screen' ? 'screen-video' : 'camera-video';
                        const videoElement = document.getElementById(videoId);
                        
                        if (videoElement) {
                            videoElement.srcObject = new MediaStream([track]);
                            videoElement.classList.remove('hidden');
                            videoElement.play().catch(console.error);
                        }
                    } else if (track.kind === 'audio') {
                        // Handle audio track
                        const audioStream = new MediaStream([track]);
                        const audioElement = new Audio();
                        audioElement.srcObject = audioStream;
                        audioElement.play().catch(console.error);
                    }
                };
            }

            await pc.setRemoteDescription(new RTCSessionDescription(data.offer));
            const answer = await pc.createAnswer();
            await pc.setLocalDescription(answer);

            socket.emit('answer', {
                answer: answer,
                to: data.from
            });
        } catch (err) {
            console.error('Error handling offer:', err);
        }
    }
});

            
          socket.on('viewer_joined', async function(data) {
    if (isBroadcaster) {
        try {
            const pc = new RTCPeerConnection({
                iceServers: [
                    { urls: 'stun:stun.l.google.com:19302' }
                ]
            });
            
            peerConnections[data.viewer_id] = pc;
            pc.pendingCandidates = [];

            pc.onicecandidate = event => {
                if (event.candidate) {
                    socket.emit('ice_candidate', {
                        candidate: event.candidate,
                        to: data.viewer_id
                    });
                }
            };

            // Add all active streams
            for (const [type, stream] of Object.entries(streams)) {
                if (stream) {
                    stream.getTracks().forEach(track => {
                        pc.addTrack(track, stream);
                    });
                }
            }

            // Create and send offer only if we don't have a local description
            if (pc.localDescription === null) {
                const offer = await pc.createOffer();
                await pc.setLocalDescription(offer);
                
                socket.emit('offer', {
                    offer: offer,
                    to: data.viewer_id
                });
            }

        } catch (err) {
            console.error('Error creating peer connection:', err);
        }
    }
});


socket.on('answer', async function(data) {
    if (isBroadcaster) {
        try {
            const pc = peerConnections[data.from];
            if (pc && pc.remoteDescription === null) {
                await pc.setRemoteDescription(new RTCSessionDescription(data.answer));
            }
        } catch (err) {
            console.error('Error handling answer:', err);
        }
    }
});
        </script>
    </body>
    </html>
    '''

@socketio.on('request_share')
def handle_share_request(data):
    room_id = data['room']
    if room_id in rooms:
        emit('share_request', {
            'from_sid': request.sid,
            'display_name': rooms[room_id]['viewers'][request.sid]['display_name']
        }, room=rooms[room_id]['broadcaster'])

@socketio.on('share_response')
def handle_share_response(data):
    emit('share_request_response', {
        'approved': data['approved']
    }, room=data['target_sid'])

@socketio.on('kick_user')
def handle_kick_user(data):
    room_id = data['room']
    target_sid = data['target_sid']
    
    if room_id in rooms and request.sid == rooms[room_id]['broadcaster']:
        emit('kicked', room=target_sid)
        leave_room(room_id, sid=target_sid)
        if target_sid in rooms[room_id]['viewers']:
            display_name = rooms[room_id]['viewers'][target_sid]['display_name']
            rooms[room_id]['viewers'].pop(target_sid)
            
            emit('message', {
                'type': 'system',
                'message': f"{display_name} has been kicked from the room"
            }, room=room_id)
            
            emit('user_list_updated', {
                'users': rooms[room_id]['viewers']
            }, room=room_id)

@socketio.on('ban_user')
def handle_ban_user(data):
    room_id = data['room']
    target_sid = data['target_sid']
    
    if room_id in rooms and request.sid == rooms[room_id]['broadcaster']:
        rooms[room_id]['banned_users'].add(request.remote_addr)  # Or other identifier
        emit('banned', room=target_sid)
        leave_room(room_id, sid=target_sid)
        if target_sid in rooms[room_id]['viewers']:
            display_name = rooms[room_id]['viewers'][target_sid]['display_name']
            rooms[room_id]['viewers'].pop(target_sid)
            
            emit('message', {
                'type': 'system',
                'message': f"{display_name} has been banned from the room"
            }, room=room_id)
            
            emit('user_list_updated', {
                'users': rooms[room_id]['viewers']
            }, room=room_id)

@socketio.on('toggle_mute')
def handle_toggle_mute(data):
    room_id = data['room']
    target_sid = data['target_sid']
    
    if room_id in rooms and request.sid == rooms[room_id]['broadcaster']:
        if target_sid in rooms[room_id]['viewers']:
            is_muted = target_sid in rooms[room_id]['muted_users']
            if is_muted:
                rooms[room_id]['muted_users'].remove(target_sid)
            else:
                rooms[room_id]['muted_users'].add(target_sid)
            
            display_name = rooms[room_id]['viewers'][target_sid]['display_name']
            emit('message', {
                'type': 'system',
                'message': f"{display_name} has been {'unmuted' if is_muted else 'muted'}"
            }, room=room_id)
            
            rooms[room_id]['viewers'][target_sid]['muted'] = not is_muted
            emit('user_list_updated', {
                'users': rooms[room_id]['viewers']
            }, room=room_id)

@socketio.on('create_room')
def handle_create_room(data):
    room_id = data['room_id']
    if room_id in rooms:
        emit('room_error', {'message': 'Room already exists'})
        return
    
    rooms[room_id] = {
        'password': data['password'],
        'broadcaster': request.sid,
        'viewers': {
            request.sid: {
                'display_name': data['display_name'],
                'isBroadcaster': True
            }
        },
        'messages': [],
        'banned_users': set(),
        'muted_users': set(),
        'active_streams': {  # Initialize the active_streams dictionary
            'camera': False,
            'screen': False,
            'mic': False
        }
    }
    
    join_room(room_id)
    system_message = f"Room created by {data['display_name']}"
    rooms[room_id]['messages'].append({'type': 'system', 'message': system_message})
    
    emit('room_joined', {
        'messages': rooms[room_id]['messages'],
        'users': rooms[room_id]['viewers']
    })
    
    emit('message', {
        'type': 'system',
        'message': system_message
    }, room=room_id)

@socketio.on('join_room')
def handle_join_room(data):
    room_id = data['room_id']
    if room_id not in rooms:
        emit('room_error', {'message': 'Room does not exist'})
        return
    
    if rooms[room_id]['password'] != data['password']:
        emit('room_error', {'message': 'Invalid password'})
        return
    
    rooms[room_id]['viewers'][request.sid] = {
        'display_name': data['display_name'],
        'isBroadcaster': False
    }
    
    join_room(room_id)
    
    # Send room history and user list to new viewer
    emit('room_joined', {
        'messages': rooms[room_id]['messages'],
        'users': rooms[room_id]['viewers']
    })
    
    # Notify broadcaster of new viewer
    emit('viewer_joined', {
        'viewer_id': request.sid
    }, room=rooms[room_id]['broadcaster'])
    
    # Notify all users of new viewer
    system_message = f"{data['display_name']} joined the room"
    rooms[room_id]['messages'].append({'type': 'system', 'message': system_message})
    
    emit('message', {
        'type': 'system',
        'message': system_message
    }, room=room_id)
    
    emit('user_list_updated', {
        'users': rooms[room_id]['viewers']
    }, room=room_id)

@socketio.on('message')
def handle_message(data):
    room = data['room']
    if room in rooms:
        message_data = {
            'type': 'chat',
            'display_name': data['display_name'],
            'message': data['message'],
            'timestamp': datetime.now().strftime('%H:%M:%S')
        }
        rooms[room]['messages'].append(message_data)
        emit('message', message_data, room=room)

@socketio.on('offer')
def handle_offer(data):
    emit('offer', {
        'offer': data['offer'],
        'from': request.sid
    }, room=data['to'])

@socketio.on('answer')
def handle_answer(data):
    emit('answer', {
        'answer': data['answer'],
        'from': request.sid
    }, room=data['to'])

@socketio.on('ice_candidate')
def handle_ice_candidate(data):
    emit('ice_candidate', {
        'candidate': data['candidate'],
        'from': request.sid
    }, room=data['to'])

@socketio.on('disconnect')
def handle_disconnect():
    for room_id, room_info in rooms.items():
        if request.sid in room_info['viewers']:
            display_name = room_info['viewers'][request.sid]['display_name']
            is_broadcaster = room_info['viewers'][request.sid]['isBroadcaster']
            
            if is_broadcaster:
                # If broadcaster disconnects, close the room
                system_message = f"Broadcast ended by {display_name}"
                emit('message', {
                    'type': 'system',
                    'message': system_message
                }, room=room_id)
                rooms.pop(room_id)
                break
            else:
                # If viewer disconnects, remove them from the room
                room_info['viewers'].pop(request.sid)
                system_message = f"{display_name} left the room"
                room_info['messages'].append({'type': 'system', 'message': system_message})
                
                emit('message', {
                    'type': 'system',
                    'message': system_message
                }, room=room_id)
                
                emit('user_list_updated', {
                    'users': room_info['viewers']
                }, room=room_id)
                break

@socketio.on('stream_status')
def handle_stream_status(data):
    room_id = data['room']
    if room_id in rooms:
        rooms[room_id]['active_streams'][data['type']] = data['active']
        # Notify all users about stream status change
        emit('stream_status_changed', {
            'type': data['type'],
            'active': data['active']
        }, room=room_id)


if __name__ == '__main__':
    socketio.run(app, host='0.0.0.0', port=5000, debug=True)